"""
title: Dialectical Synthesis Pipeline
author: Michael Böhm, Cephei AG
version: 1.4
description: These-Antithese-Synthese mit zwei Modellen (Streaming) + Websuche für Kritiker
requirements: requests
"""

from typing import List, Union, Generator, Iterator
from pydantic import BaseModel
import requests
import json

class Pipeline:
    class Valves(BaseModel):
        OPENAI_API_BASE: str = "http://localhost:8080/api/v1"
        MODEL_THESIS: str = "gemma3:27b"
        MODEL_ANTITHESIS: str = "gpt-oss:20b"
        API_KEY: str = "sk-withheld"
        SEARXNG_URL: str = "http://localhost:12790/search"
        BRAVE_API_KEY: str = "" # nur für alternative Suchfunktion mit Brave
        SEARCH_RESULTS_COUNT: int = 25

    def __init__(self):
        self.name = "Dialektische Synthese"
        self.valves = self.Valves()

    def search_web(self, query: str) -> str:
        """Führt eine Websuche durch und gibt formatierte Ergebnisse zurück"""
        try:
            response = requests.get(
                self.valves.SEARXNG_URL,
                params={"q": query, "format": "json"},
                timeout=10
            )
            response.raise_for_status()
            data = response.json()
            results = data.get("results", [])[:self.valves.SEARCH_RESULTS_COUNT]

            if not results:
                return "Keine Suchergebnisse gefunden."

            formatted = []
            for r in results:
                title = r.get("title", "Ohne Titel")
                url = r.get("url", "")
                content = r.get("content", "")[:200]
                formatted.append(f"- **{title}**\n  URL: {url}\n  {content}")

            return "\n\n".join(formatted)
        except Exception as e:
            return f"Websuche fehlgeschlagen: {str(e)}"

    def stream_model(self, model: str, messages: List[dict]) -> Generator:
        """Streamt die Antwort eines Modells"""
        response = requests.post(
            f"{self.valves.OPENAI_API_BASE}/chat/completions",
            headers={"Authorization": f"Bearer {self.valves.API_KEY}"},
            json={
                "model": model,
                "messages": messages,
                "stream": True
            },
            stream=True
        )

        full_response = ""
        for line in response.iter_lines():
            if line:
                line = line.decode('utf-8')
                if line.startswith('data: '):
                    data = line[6:]
                    if data == '[DONE]':
                        break
                    try:
                        chunk = json.loads(data)
                        if 'choices' in chunk and len(chunk['choices']) > 0:
                            delta = chunk['choices'][0].get('delta', {})
                            content = delta.get('content', '')
                            if content:
                                full_response += content
                                yield content
                    except json.JSONDecodeError:
                        pass

        return full_response

    def call_model(self, model: str, messages: List[dict]) -> str:
        """Ruft ein Modell auf und gibt die komplette Antwort zurueck"""
        response = requests.post(
            f"{self.valves.OPENAI_API_BASE}/chat/completions",
            headers={"Authorization": f"Bearer {self.valves.API_KEY}"},
            json={
                "model": model,
                "messages": messages,
                "stream": False
            }
        )
        return response.json()["choices"][0]["message"]["content"]

    def fix_headings(self, text: str) -> str:
        """Ersetzt grosse Ueberschriften durch kleinere"""
        lines = text.split('\n')
        fixed_lines = []
        for line in lines:
            if line.startswith('# ') and not line.startswith('## ') and not line.startswith('### '):
                line = '###' + line[1:]
            elif line.startswith('## ') and not line.startswith('### '):
                line = '###' + line[2:]
            fixed_lines.append(line)
        return '\n'.join(fixed_lines)

    def pipe(
        self, user_message: str, model_id: str, messages: List[dict], body: dict
    ) -> Generator:

        # Filtere alte Pipeline-Überschriften aus den Messages (werden sonst manchmal verdoppelt)
        cleaned_messages = []
        for m in messages:
            content = m.get("content", "")
            # Entferne Pipeline-Formatierung aus alten Assistant-Antworten
            if m.get("role") == "assistant" and "## 💭 These" in content:
                # Extrahiere nur die Synthese (letzter Teil nach "## ✅ Synthese")
                if "## ✅ Synthese" in content:
                    parts = content.split("## ✅ Synthese")
                    if len(parts) > 1:
                        # Nimm nur den Synthese-Teil, bereinigt
                        synthesis = parts[-1].split("> 🟢 **Verbesserte Antwort**\n>\n")[-1]
                        cleaned_messages.append({"role": "assistant", "content": synthesis.strip()})
                        continue
            cleaned_messages.append(m)

        messages = cleaned_messages

        # ============ SCHRITT 1: THESE ============
        yield f"## 💭 These (Modell: {self.valves.MODEL_THESIS})\n\n"
        yield "> 🔵 **Erste Analyse**\n>\n"

        thesis_messages = [
            {
                "role": "system",
                "content": "Du bist ein hilfreicher Assistent. Antworte immer in natuerlicher Sprache mit Markdown-Formatierung. Gib NIEMALS JSON, XML oder andere strukturierte Datenformate aus. Verwende Ueberschriften (###), Listen und Fliesstext."
            }
        ] + messages

        thesis = ""
        for chunk in self.stream_model(self.valves.MODEL_THESIS, thesis_messages):
            thesis += chunk
            yield self.fix_headings(chunk)

        yield "\n\n---\n\n"

        # ============ SCHRITT 2: SUCHBEGRIFFE VOM KRITIKER ============
        yield f"## 🌐 Faktencheck via Websuche\n\n"
        yield "> 🟣 **Kritiker generiert Suchbegriffe...**\n>\n"

        search_query_prompt = f"""Du wirst gleich die folgende Antwort kritisch pruefen. Zuerst brauchst du aktuelle Informationen aus dem Web.

**Urspruengliche Frage des Nutzers:** {user_message}

**Zu pruefende Antwort:**
{thesis}

Welche 2-3 Suchbegriffe wuerdest du verwenden, um die Fakten in dieser Antwort zu verifizieren?
Die Begriffe sollen allgemein genug sein, um gute Treffer zu liefern.

Antworte NUR mit den Suchbegriffen, getrennt durch Komma. Keine Erklaerung, kein anderer Text.
Beispiel-Antwort: Open WebUI API key, Open WebUI settings"""

        search_query_messages = [{"role": "user", "content": search_query_prompt}]
        search_query = self.call_model(self.valves.MODEL_ANTITHESIS, search_query_messages).strip()

        # Falls das Modell zu viel Text liefert, nur die erste Zeile nehmen
        search_query = search_query.split('\n')[0][:200]

        yield f"> *Suchbegriffe: {search_query}*\n>\n"

        # ============ SCHRITT 3: WEBSUCHE ============
        search_results = self.search_web(search_query)
        yield search_results
        yield "\n\n---\n\n"

        # ============ SCHRITT 4: ANTITHESE ============
        yield f"## 🔍 Kritische Prüfung (Modell: {self.valves.MODEL_ANTITHESIS})\n\n"
        yield "> 🔴 **Kritik & Faktencheck**\n>\n"

        antithesis_prompt = f"""Du bist ein kritischer Reviewer. Du hast soeben folgende Suchbegriffe gewaehlt: {search_query}

**Urspruengliche Frage:** {user_message}

**Zu pruefende Antwort:**
{thesis}

**Deine Websuche-Ergebnisse:**
{search_results}

Analysiere die Antwort auf:
- Faktische Fehler (vergleiche mit den Websuche-Ergebnissen!)
- Schwaechen und fehlende Aspekte
- Veraltete Informationen
- Moegliche Verbesserungen

FORMATIERUNG:
- Verwende KEINE grossen Ueberschriften (kein # oder ##)
- Nutze nur ### oder #### fuer Abschnitte
- Beginne NICHT mit einer Ueberschrift, sondern direkt mit Text
- Beispiel fuer erlaubte Ueberschrift: ### Faktische Fehler

Gib eine strukturierte Kritik mit konkreten Verbesserungsvorschlaegen."""

        antithesis_messages = [{"role": "user", "content": antithesis_prompt}]

        antithesis = ""
        for chunk in self.stream_model(self.valves.MODEL_ANTITHESIS, antithesis_messages):
            antithesis += chunk
            yield self.fix_headings(chunk)

        yield "\n\n---\n\n"

        # ============ SCHRITT 5: SYNTHESE ============
        yield f"## ✅ Synthese (Modell: {self.valves.MODEL_THESIS})\n\n"
        yield "> 🟢 **Verbesserte Antwort**\n>\n"

        synthesis_prompt = f"""Du hast folgende Antwort gegeben:

{thesis}

Ein kritischer Reviewer hat diese Punkte angemerkt (basierend auf einer Websuche mit den Begriffen: {search_query}):

{antithesis}

Erstelle nun eine verbesserte, finale Synthese, die:
- Deine urspruengliche Antwort mit den berechtigten Kritikpunkten vereint
- Faktische Fehler korrigiert (beachte die Websuche-Ergebnisse!)
- Fehlende Aspekte ergaenzt
- Die Staerken beider Perspektiven kombiniert

FORMATIERUNG:
- Verwende KEINE grossen Ueberschriften (kein # oder ##)
- Nutze nur ### oder #### fuer Abschnitte
- Beginne NICHT mit einer Ueberschrift, sondern direkt mit Text
- Antworte in natuerlicher Sprache, KEIN JSON oder XML."""

        synthesis_messages = [
            {
                "role": "system",
                "content": "Du bist ein hilfreicher Assistent. Antworte immer in natuerlicher Sprache mit Markdown-Formatierung. Gib NIEMALS JSON, XML oder andere strukturierte Datenformate aus."
            }
        ] + messages
        synthesis_messages.append({"role": "assistant", "content": thesis})
        synthesis_messages.append({"role": "user", "content": synthesis_prompt})

        for chunk in self.stream_model(self.valves.MODEL_THESIS, synthesis_messages):
            yield self.fix_headings(chunk)
