#!/usr/bin/env python3

import argparse, math, secrets, string, sys

try:
    import pyperclip
except ModuleNotFoundError:
    pyperclip = None

DEFAULT_LENGTH = 16
VERSION = "1.0.0"
MAX_LENGTH = 2048
ALLOWED_SPECIAL = "!@#$%*()_-"  # may not be empty
AMBIGUOUS_CHARS = set("0O1lI")
UPPERCASE_LETTERS = "".join(char for char in string.ascii_uppercase if char not in AMBIGUOUS_CHARS)
LOWERCASE_LETTERS = "".join(char for char in string.ascii_lowercase if char not in AMBIGUOUS_CHARS)
DIGITS = "".join(char for char in string.digits if char not in AMBIGUOUS_CHARS)

def get_charset(include_special: bool = True) -> str:
    """
    Get the full character set based on whether specials are included.

    Args:
        include_special: Whether special characters are allowed.

    Returns:
        The combined character set string.
    """
    allowed_special = ALLOWED_SPECIAL if include_special else ""
    return UPPERCASE_LETTERS + LOWERCASE_LETTERS + DIGITS + allowed_special

def estimate_entropy_bits(length: int, include_special: bool = True) -> float:
    """
    Estimate password entropy in bits based on the actual generation algorithm.
    The calculation accounts for the guaranteed character requirements:
    - At least one uppercase letter
    - At least one lowercase letter
    - At least one digit
    - At least one special character (if include_special=True)
    - Remaining positions filled from full charset

    Args:
        length: Desired password length.
        include_special: Whether special characters are allowed.

    Returns:
        Estimated entropy in bits.
    """
    minimum_length = 4 if include_special else 3
    if length < minimum_length:
        raise ValueError(f"length must be at least {minimum_length}")

    charset = get_charset(include_special=include_special)
    charset_size = len(charset)
    
    # Entropy from guaranteed characters (constrained to subsets)
    entropy = math.log2(len(UPPERCASE_LETTERS))
    entropy += math.log2(len(LOWERCASE_LETTERS))
    entropy += math.log2(len(DIGITS))
    
    if include_special:
        entropy += math.log2(len(ALLOWED_SPECIAL))
        guaranteed_count = 4
    else:
        guaranteed_count = 3
    
    # Entropy from remaining positions (full charset)
    remaining = length - guaranteed_count
    entropy += remaining * math.log2(charset_size)

    return entropy

def generate_password(length: int, include_special: bool = True) -> str:
    """
    Generate a random password of the specified length.

    Args:
        length: Desired password length (must be >= 4 with specials, >= 3 without,
            and <= MAX_LENGTH).
        include_special: Whether to include special characters in the password.

    Returns:
        A random password string.

    Raises:
        ValueError: If length is outside the valid range (minimum or maximum).
    """
    minimum_length = 4 if include_special else 3
    if length < minimum_length:
        raise ValueError(f"length must be at least {minimum_length}")
    if length > MAX_LENGTH:
        raise ValueError(f"length must not exceed {MAX_LENGTH}")

    all_chars = get_charset(include_special=include_special)

    password_chars = [
        secrets.choice(UPPERCASE_LETTERS),
        secrets.choice(LOWERCASE_LETTERS),
        secrets.choice(DIGITS),
    ]
    if include_special:
        password_chars.append(secrets.choice(ALLOWED_SPECIAL))

    remaining_length = length - len(password_chars)
    password_chars.extend([secrets.choice(all_chars) for _ in range(remaining_length)])

    secrets.SystemRandom().shuffle(password_chars)

    return "".join(password_chars)

def main() -> None:
    parser = argparse.ArgumentParser(
        description="Generate a random password of a given length.",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  python genpwd.py 12                   # Generate a 12-character password
  python genpwd.py                      # Generate a default length password
  python genpwd.py 12 --no-special      # Generate without special characters
  python genpwd.py 16 --no-clipboard    # Print password without copying
  python genpwd.py 20 --verbose         # Show estimated entropy in bits
        """,
    )
    parser.add_argument(
        "length",
        type=int,
        nargs="?",
        default=DEFAULT_LENGTH,
        help=f"Password length (default: {DEFAULT_LENGTH}, max: {MAX_LENGTH})",
    )
    parser.add_argument(
        "-s",
        "--no-special",
        action="store_true",
        dest="no_special",
        help="Exclude special characters from the password",
    )
    parser.add_argument(
        "-n",
        "--no-clipboard",
        action="store_true",
        help="Do not copy the password to the clipboard",
    )
    parser.add_argument("-v", "--version", action="version", version=f"%(prog)s {VERSION}")
    parser.add_argument(
        "-b",
        "--verbose",
        action="store_true",
        help="Show estimated password entropy in bits",
    )

    args = parser.parse_args()

    try:
        password = generate_password(args.length, include_special=not args.no_special)
    except ValueError as exc:
        print(f"Error: {exc}", file=sys.stderr)
        sys.exit(1)

    if not args.no_clipboard:
        if pyperclip is None:
            print(
                "pyperclip not installed; copy skipped (--no-clipboard to suppress this message)",
                file=sys.stderr,
            )
        else:
            try:
                pyperclip.copy(password)
            except Exception:
                print("Could not copy to clipboard", file=sys.stderr)

    if args.verbose:
        entropy_bits = estimate_entropy_bits(args.length, include_special=not args.no_special)
        print(f"Estimated entropy: {entropy_bits:.2f} bits", file=sys.stderr)

    print(password)

if __name__ == "__main__":
    main()
